//*************************************************************************************************
//
//	Description:
//		glass.fx - HQ glass shader for use with cockpit glass & other glassy objects close to the
//							 camera. Supports reflection and mapping, fresnel terms, non-glass fragments,
//							 special effects masks, damage textures, dirt textures and water textures.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author:
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     07/13/2007  0.1           Created
//		TNettleship     07/24/2007  0.2           Made sure samplers aren't using anisotropic filtering.
//		TNettleship     10/04/2007  0.3           Improved envmapping, fixed brightness problem.
//		TNettleship     10/08/2007  0.4           Total redesign; now uses just one combined layer.
//		TNettleship			10/23/2007	0.5						Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// We use preprocessor definitions instead of variables in this shader, to ensure consistent glass values across all cars
#define GLOBAL_ENVMAP_FACTOR 1.0f
#define GLOBAL_SPECULAR_FACTOR 4.0f
#define GLOBAL_SPECULAR_POWER 8192.0f

#define MAX_ENVMAP_BIAS	6.0f

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define NORMAL_MAPPING
#define REFLECTION_MAPPING
#define DIRT_SCRATCH
#define USE_VINYLS
#define USE_VINYLS_BACKSIDE
#define ALLOW_VINYLS
#endif

#ifdef DIRT_SCRATCH
//#define SPECIAL_WRAPPING		// used to provide a visualisation of the data when nothing seems to work - please do not remove related code as it is very useful for testing
#endif

// Uncomment this to enable the parralax effect on reflection maps (not used in current titles)
#define ENABLE_PARRALAX_REFLECTION

// Uncomment to use experimental higher quality envmapping approach
#define EXPERIMENTAL_ENVMAPPING


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else		// ! _3DSMAX_
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif	// ! _3DSMAX_



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else		// ! _3DSMAX_ && ! _PS3_
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif	// ! _3DSMAX_ && ! _PS3_

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;


// This matrix is only needed if envmapping is being used, or dirt and scratches are enabled
#if !defined( REFLECTION_MAPPING ) || defined( DIRT_SCRATCH )

#if defined( _3DSMAX_ )
float4x4 worldI : WorldInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
float4x4 envTransformI : EnvInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
float4x4 envTransformI : EnvInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif

#endif	// ! REFLECTION_MAPPING || DIRT_SCRATCH



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//
#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Reflection mapping UVs. Exported as texcoord1, when required.
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Third UV channel. N.B. Uses mapchannel 4 to match bodywork.fx
int texcoord4 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 4;
	int MapChannel = 4;
	int RuntimeTexcoord = 3;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif	// _3DSMAX_



//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

SPECIALISATION_PARAM_DEFAULTS_TRUE( useSpecularMap, "Use specular map?", "SPECULAR_MAPPING" )
#if defined( _3DSMAX_ ) || defined( SPECULAR_MAPPING )
DEPENDENT_TEXTURE_PARAM( specularTexture, "Spec Tex {UV1}", useSpecularMap )		// Specular colour in RGB, glossiness in alpha
#endif

float minSpecPower
<
	string UIName = "Min Spec";
	float UIMin = 1.0f;
	float UIMax = 65536.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Spec";
	float UIMin = 1.0f;
	float UIMax = 65536.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Spec Factor";
	float UIMin = 0.0f;
	float UIMax = 16.0f;
	bool appEdit = true;
> = 1.0f;

SPECIALISATION_PARAM( useFresnelMap, "Use fresnel map?", "FRESNEL_MAPPING" )
#if defined( _3DSMAX_ ) || defined( FRESNEL_MAPPING )
DEPENDENT_TEXTURE_PARAM( fresnelTexture, "Fresnel Tex {UV1}", useFresnelMap )		// Fresnel control texture, green 0.0 = no fresnel, green 1.0 = full fresnel
#endif

float fresnelFactor
<
	string UIName = "Fresnel Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 0.6666f;

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
DEPENDENT_BOOL_PARAM( specialNormalMap, "Special normal map format?", useNormalMap )
#endif

SPECIALISATION_PARAM( useReflectionMap, "Use refl. map?", "REFLECTION_MAPPING" )
DECLARE_DEPENDENT_VERTEX_STREAM( uv1Dependency, texCoord1, TEXCOORD2, useReflectionMap )

#if defined( _3DSMAX_ ) || defined( REFLECTION_MAPPING )
DEPENDENT_TEXTURE_PARAM( reflectionTexture, "Refl. Tex {UV2}", useReflectionMap )

#if defined( ENABLE_PARRALAX_REFLECTION )
// Params for reflection parralax effect
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxUOffset, "Max U Offset", useReflectionMap, 0.0f, 1.0f, 0.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxVOffset, "Max V Offset", useReflectionMap, 0.0f, 1.0f, 0.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( headUMovement, "Head U Movement", useReflectionMap, -1.0f, 1.0f, 0.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( headVMovement, "Head V Movement", useReflectionMap, -1.0f, 1.0f, 0.0f )
#endif	// defined( ENABLE_PARRALAX_REFLECTION )

#endif	// _3DSMAX_ || REFLECTION_MAPPING


// Vinyls (custom livery).
//
// Note that vinyl texture uses {UV4} Max channel, sharing it with dirt'n'scratches and reflection mapping.
// Thus either one of these features (USE_VINYLS, ALLOW_VINYLS, DIRT_SCRATCH) is enough to export {UV4}.

SPECIALISATION_PARAM( useVinyls, "Use vinyls?", "USE_VINYLS" )
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
DEPENDENT_TEXTURE_PARAM( vinylsTexture, "Vinyls Tex {UV4}", useVinyls )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vinylsSpecularFactor, "Specular Factor of Vinyls", useVinyls, 0.0f, 16.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vinylsSpecularPower, "Specular Power of Vinyls (min/max)", useVinyls, 0.0f, 65536.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vinylsEMapFactor, "EMap Factor of Vinyls", useVinyls, 0.0f, 16.0f, 0.7f )
#endif
DECLARE_DEPENDENT_VERTEX_STREAM( uv2VinylsDependency, texCoord2, TEXCOORD4, useVinyls )

SPECIALISATION_PARAM( allowVinyls, "Allow vinyls?", "ALLOW_VINYLS" )
DECLARE_DEPENDENT_VERTEX_STREAM( uv2VinylsDependency2, texCoord2, TEXCOORD4, allowVinyls )

// Use the reverse order of layers, to simulate vinyls placed on the outer surface of glass.
// Used for cockpits; makes sense only if USE_VINYLS or ALLOW_VINYLS is on.

SPECIALISATION_PARAM( useVinylsBackside, "Use backside of vinyls?", "USE_VINYLS_BACKSIDE" )

#if defined( _3DSMAX_ ) || ( defined( USE_VINYLS ) && defined( USE_VINYLS_BACKSIDE ) )
BOTH_DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( vinylsBacksideColour,  "Backside Colour Vinyls",  useVinyls;useVinylsBackside, 0.2f, 0.2f, 0.2f, 0.8f  )
BOTH_DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vinylsBackviewOpacity, "Maximum opacity of Vinyls from backside", useVinyls;useVinylsBackside, 0.0f, 1.0f, 0.7f )
#endif


// Constant diffuse colour
//
// Use constant diffuse, as opposed to one delivered via diffuse texture.
// Used for run-time tinting of vehicle windows.

SPECIALISATION_PARAM( useConstantDiffuse, "Use constant diffuse colour?", "USE_CONSTANT_DIFFUSE" )
#if defined( _3DSMAX_ ) || defined( USE_CONSTANT_DIFFUSE )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( constantDiffuse, "Constant Diffuse Colour", useConstantDiffuse, 0.1f, 0.1f, 0.1f, 0.45f )
#endif


#if defined( _3DSMAX_ ) || !defined( REFLECTION_MAPPING ) || defined( DIRT_SCRATCH )
// Environment

texture environmentTexture : TEXTURE					// Environment map (RGB)
<
	string UIName = "Env Texture";
	string Type = "CUBE";
	bool appEdit = false;
	bool export = false;
>;

#endif


float globalEMapFactor
<
	string UIName = "Global Emap Factor";
	float UIMin = 0.0f;
	float UIMax = 16.0f;
	bool appEdit = true;
> = 1.0f;


//
// Dirt n scratches
//

SPECIALISATION_PARAM( dirtNscratch, "Dirt 'n' Scratch?", "DIRT_SCRATCH" )	// TRUE if dirt and scratches should be applied
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDirtDependency, tangent, TANGENT, dirtNscratch )			// second dependency for these
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDirtDependency, binormal, BINORMAL, dirtNscratch )		// second dependency for these
DECLARE_DEPENDENT_VERTEX_STREAM( uv2DirtDependency, texCoord2, TEXCOORD4, dirtNscratch )			// second dependency for these

#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
DEPENDENT_TEXTURE_PARAM( scratchControlTexture, "Dirt / Scratch Tex.", dirtNscratch )			// multi-channel dirt and scratch texture, will be supplied by game
DEPENDENT_TEXTURE_PARAM( crackTexture, "Crack Texture", dirtNscratch )										// special bidirectional normal map for cracks
DEPENDENT_BOOL_PARAM( specialCrackNormalMap, "Special normal map format?", dirtNscratch )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( crackColour, "Crack Colour", dirtNscratch, 0.7f, 0.7f, 0.7f, 1.0f  )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( dirtBasis,  "Dirt Colour",  dirtNscratch, 0.233f, 0.203f, 0.100f, 1.0f  )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( noiseScale, "Dirt Noise Scale", dirtNscratch, 0.0f, 128.0f, 4.0f )
DEPENDENT_BOOL_PARAM( internalView, "Interior glass?", dirtNscratch)	// used to control how opacity works and effect of dirt on reflectiveness
#if defined( _3DSMAX_ )	// this one is only for Max testing, so excluded from game builds
DEPENDENT_BOOL_PARAM( testMode, "Test cracks", dirtNscratch)	// used to fake damage at full value everywhere for cracks
#endif
DEPENDENT_FLOAT4_PARAM_DEFAULTED( crackUVoffset, crackUVoffsetX, crackUVoffsetY, crackUVoffsetZ, crackUVoffsetW, "Crack U offset", "Crack V offset", "Crack scale U", "Crack scale V", dirtNscratch, 0.0f, 0.0f, 1.0f, 1.0f  )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( crackThickness,		"Crack Thickness Scale", dirtNscratch, 0.0f, 16.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( sensitivity,				"Sensitivity to texture", dirtNscratch, 0.0f, 1.0f, 0.7f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( rangeVisibility,		"Scale to allow for distance", dirtNscratch, 0.0f, 20.0f, 5.0f )

DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( crackLineScale,		"Scale for thinner lines", dirtNscratch, 0.0f, 1.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( crackStrength,			"Multiplier used on cracks", dirtNscratch, 1.0f, 100.0f, 20.0f )

DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( damageStartCracks, "Damage level at which cracks appear", dirtNscratch, 0.0f, 1.0f, 0.25f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( damageCrackSlope,	"Rate cracks increase with damage", dirtNscratch, 0.0f, 100.0f, 10.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( crackFinalScale,		"Scale for final output", dirtNscratch, 0.0f, 2.0f, 1.0f )
#endif


// Simplify shadowing code to access only the first (the nearest) shadowmap.
SPECIALISATION_PARAM( forceFistShadowmap, "Force the first shadowmap?", "FORCE_FIRST_SHADOWMAP" )


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( SPECULAR_MAPPING )
sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif	// _3DSMAX_ || SPECULAR_MAPPING

#if defined( _3DSMAX_ ) || defined( FRESNEL_MAPPING )
sampler2D fresnelMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="fresnelTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < fresnelTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif	// _3DSMAX_ ) || FRESNEL_MAPPING

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif	// _3DSMAX_ || NORMAL_MAPPING

#if defined( _3DSMAX_ ) || !defined( REFLECTION_MAPPING ) || defined( DIRT_SCRATCH )
// This sampler is only needed if envmapping is being used
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif	// _3DSMAX_ || ! REFLECTION_MAPPING || DIRT_SCRATCH

#if defined( _3DSMAX_ ) || defined( REFLECTION_MAPPING )
sampler2D reflectionMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="reflectionTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < reflectionTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif	// _3DSMAX_ || REFLECTION_MAPPING

#if defined( _3DSMAX_ ) || defined ( DIRT_SCRATCH )
sampler2D scratchControlMap : sample
<
	SET_LINEAR_TEXTURE
	bool appEdit = false;
	string SamplerTexture="scratchControlTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < scratchControlTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
#endif
};

sampler2D crackMap : sample
<
	SET_LINEAR_TEXTURE
	bool appEdit = false;
	string SamplerTexture="crackTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < crackTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
#endif
};
#endif	// _3DSMAX_ || DIRT_SCRATCH

#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
sampler2D vinylsMap : SAMPLER
<
	SET_SRGB_TEXTURE
	bool appEdit = false;
	string SamplerTexture = "vinylsTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < vinylsTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif	// _3DSMAX_ || USE_VINYLS


//
// Functions
//


//
// This seems to work almost as well as the full-on "complicated fresnel"
// A good rindexRatio for glass is 0.6667 (which assumes air = 1.0, glass = 1.5)
//
float calculated_fresnel(float3 _eye, float3 _normal, float rindexRatio)
{
	// Note: compute R0 on the CPU and provide as a
	// constant; it is more efficient than computing R0 in
	// the vertex shader. R0 is:
	// float const R0 = pow(1.0-refractionIndexRatio, 2.0)
	// / pow(1.0+refractionIndexRatio, 2.0);
	// light and normal are assumed to be normalized
	float R0 = pow(1.0f - rindexRatio, 2.0f) / pow ( 1.0f + rindexRatio, 2.0f );
//	float R0 = ( 1.0f - rindexRatio ) / ( 1.0f + rindexRatio );
//	float R0 = ( 2.0f / ( 1.0f + rindexRatio ) ) - 1.0f;
//	R0 = R0 * R0;

	return R0 + (1.0f-R0) * pow(1.0f - saturate( dot(-_eye, _normal) ), 5.0f);
}


//-----------------------------------------------------------------------
//## OpacityOfCompound()
//
//  Description:
//    Given opacities of two layers, compute opacity of the compound.
//
//    "One layer lets 40% percent of light pass through, the other 30% of the remainder."
//
float GetOpacityOfCompound( float opacity1, float opacity2 )
{
	float translucency1 = ( 1.0f - opacity1 );
	float translucency2 = ( 1.0f - opacity2 );
	float translucency = translucency1 * translucency2;
	return ( 1.0f - translucency );
}


//-----------------------------------------------------------------------
//## Get2DUVCoordinates()
//
//  Description:
//    Given a pair of UVs, convert them into normal form - it involves flipping V in Max.
//
float2 Get2DUVCoordinates( float2 uv )
{
#if defined( _3DSMAX_ )
	return float2( uv.x, 1.0f - uv.y );
#else
	return uv;
#endif	// ! _3DSMAX_
}


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position  : POSITION;													// Object space position
#ifdef _3DSMAX_
	float3 colour    : TEXCOORD1;													// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 1 texture coord - N.B. Max requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float2 texCoord1 : TEXCOORD2;													// UV channel 2 texture coord (reflection map)
	float3 alpha		 : TEXCOORD3;													// Vertex alpha
	float2 texCoord2 : TEXCOORD4;													// UV channel 4 texture coord (special effect map / control map)
#else	// ! _3DSMAX_
	float4 colour     : COLOR0;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV channel 1 texture coord

#if defined( REFLECTION_MAPPING )
	// This vertex stream isn't needed without reflection mapping being enabled
	float2 texCoord1 : TEXCOORD1;													// UV channel 2 texture coord (reflection map)
#endif	// REFLECTION_MAPPING

#if defined( DIRT_SCRATCH ) || defined( USE_VINYLS )
	// This vertex stream isn't needed without special effects being enabled
	float2 texCoord2 : TEXCOORD3;													// UV channel 4 texture coord (special effect map / control map)
#endif	// DIRT_SCRATCH || USE_VINYLS

#endif	// ! _3DSMAX_

	float3 normal		 : NORMAL;														// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING ) || defined( DIRT_SCRATCH )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent   : TANGENT;														// Object space tangent
	float3 binormal  : BINORMAL;													// Object space normal
#endif
};



// Output structure
struct VSOUTPUT
{
	float4 position						: POSITION;						// View-coords position
	float4 colour							: TEXCOORD1;							// Vertex colour
	float4 texCoord01					: TEXCOORD0;					// UV coords for texture channel 0 (and optionally texture channel 1)

#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH ) || defined( USE_VINYLS )
	// This vertex stream isn't needed without dirt & scratch or vinyls being enabled
	float2 texCoord2					: TEXCOORD2;					// UV coords for texture channel 2
#endif

	float4 normal_and_fresnel	: TEXCOORD3;					// Normal vector (world space)
	float4 eye								: TEXCOORD4;					// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING ) || defined( DIRT_SCRATCH )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent						: TEXCOORD5;					// Tangent vector (world space)
	float3 binormal						: TEXCOORD6;					// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 5 )
#endif
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

#ifdef _POST_PARSE_
VSOUTPUT GlassVertexShader( VSINPUT _input )
#else
VSOUTPUT GlassVertexShader( VSINPUT _input )
#endif
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord01 = float4( _input.texCoord0, 0.0f, 0.0f );

	DEPENDENT_CODE_START( useReflectionMap )
#if defined( _3DSMAX_ ) || defined( REFLECTION_MAPPING )
		_output.texCoord01.zw = _input.texCoord1;
#endif
	DEPENDENT_CODE_END( useReflectionMap )

	DEPENDENT_CODE_START_OR( dirtNscratch, useVinyls )
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH ) || defined( USE_VINYLS )
		_output.texCoord2 = _input.texCoord2;
#endif
	DEPENDENT_CODE_END_OR( dirtNscratch, useVinyls )

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	float3 normal  = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	DEPENDENT_CODE_START_OR( useNormalMap, dirtNscratch )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING ) || defined( DIRT_SCRATCH )
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_END_OR( useNormalMap, dirtNscratch )

	// Calculate fresnel term
	_output.normal_and_fresnel = float4( normal, calculated_fresnel( - normalize( worldEyeVec ), normal, fresnelFactor ) );

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour *= globalColourMultiplier;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour							: TEXCOORD1;					// Vertex colour
	float4 texCoord01					: TEXCOORD0;					// UV coords for texture channel 0 (and optionally texture channel 1)
	float2 texCoord2					: TEXCOORD2;					// UV coords for texture channel 2
	float4 normal_and_fresnel	: TEXCOORD3;					// Normal vector (world space)
	float4 eye								: TEXCOORD4;					// Eye vector (world space)
	float3 tangent						: TEXCOORD5;					// Tangent vector (world space)
	float3 binormal						: TEXCOORD6;					// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
};

#else

struct PSINPUT
{
	float4 colour							: TEXCOORD1;							// Vertex colour
	float4 texCoord01					: TEXCOORD0;					// UV coords for texture channel 0 (and optionally channel 1)

#if defined( DIRT_SCRATCH ) || defined( USE_VINYLS )
	// This vertex stream isn't needed without special effects being enabled
	float2 texCoord2					: TEXCOORD2;					// UV coords for texture channel 2
#endif	// DIRT_SCRATCH || USE_VINYLS

	float4 normal_and_fresnel	: TEXCOORD3_centroid;	// Normal vector (world space)
	float4 eye								: TEXCOORD4_centroid;	// Eye vector (world space)

#if defined( NORMAL_MAPPING ) || defined( DIRT_SCRATCH )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent						: TEXCOORD5_centroid;	// Tangent vector (world space)
	float3 binormal						: TEXCOORD6_centroid;	// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 5 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif



// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};


//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT GlassFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	//
	// Read textures
	//

	float4 diffuseTexColour;
	DEPENDENT_CODE_START( useConstantDiffuse )
#if defined( _3DSMAX_ ) || defined( USE_CONSTANT_DIFFUSE )
		diffuseTexColour  = constantDiffuse;
#endif
	DEPENDENT_CODE_ELSE( useConstantDiffuse )
#if defined( _3DSMAX_ ) || ! defined( USE_CONSTANT_DIFFUSE )
		diffuseTexColour  = tex2D( diffuseMap, _input.texCoord01.xy );
#endif
	DEPENDENT_CODE_END( useConstantDiffuse )

#if defined( _3DSMAX_ ) || defined( SPECULAR_MAPPING )
	float4 specularTexColour = tex2D( specularMap, _input.texCoord01.xy );
#else
	float4 specularTexColour = float4( 1.0f, 1.0f, 1.0f, 1.0f );
#endif

#if defined( _3DSMAX_ ) || defined( FRESNEL_MAPPING )
	float  fresnelControl = tex2D( fresnelMap, _input.texCoord01.xy ).g;
#else
	float  fresnelControl = 1.0f;
#endif

#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
	float4 vinylsColour = tex2D( vinylsMap, Get2DUVCoordinates( _input.texCoord2.xy ) );
#endif

	// This factor modifies the amount of specular (and emapping) in dirty or scratchy areas. On clean glass it has no effect.
	float dirtScratchSpecularModulation = 1.0;

#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
	float4 controlColour;
	float dirtPercent;
	float dirtOpacity;
	float crackPercent;
  float2 scratchControlUV;	// used to simplify code, to reduce risk of errors
#endif	// DIRT_SCRATCH
  DECLARE_MERGED_FLOAT4_PARAM( crackUVoffset, crackUVoffsetX, crackUVoffsetY, crackUVoffsetZ, crackUVoffsetW)
	
	// Factor vertex alpha into the diffuse alpha
	float translucency = diffuseTexColour.a * _input.colour.a;

	// Premultiply the diffuse colour by the translucency - this is not effected by the mud in the interior case, hence being done first
	diffuseTexColour.rgb *= translucency;
	
//	float broken = sign(translucency);	// 0 = nothing, 1 = something

	//
	// Block (USE_VINYLS)
	//   Put a plastic layer with vinyls on top of the glass,
	//   Also prepare plenty of various other factors to be applied on various parts of shader below.
	//   Note that there is one additional USE_VINYLS block below.
	//
	float emapFactor = GLOBAL_ENVMAP_FACTOR;					// On pure glass, envmapping strength is controlled by glass material properties.
	float specularFactor = GLOBAL_SPECULAR_FACTOR;	// On pure glass, specular strength is controlled by glass material properties.
	float minSpecPowerValue = GLOBAL_SPECULAR_POWER;
	float maxSpecPowerValue = GLOBAL_SPECULAR_POWER;
	float4 vinylsEMapColour = specularTexColour;	// On pure glass, let the supplied specular property modulate reflections.
	float vinylsOpacity = 0.0f;
	float vinylsCrackScaler = 1.0f;								// On pure glass, cracks at full intensity.
	float vinylsLightingSpecularFactor = GLOBAL_SPECULAR_FACTOR;

	DEPENDENT_CODE_START_AND( useVinyls, useVinylsBackside )
#if defined( _3DSMAX_ ) || ( defined( USE_VINYLS ) && defined( USE_VINYLS_BACKSIDE ) )

		// We are looking at the vinyl from backside - adjust the colour looked up from texture toward constant.
		vinylsColour.rgb = lerp( vinylsColour.rgb, vinylsBacksideColour.rgb, vinylsBacksideColour.a );

		// And clamp the maximum opacity by another supplied constant - so a vinyl fully opaque from outside can be half-translucent from inside.
		vinylsColour.a *= vinylsBackviewOpacity;

		vinylsOpacity = vinylsColour.a;

		diffuseTexColour.rgb = lerp( diffuseTexColour.rgb, vinylsColour.rgb, vinylsOpacity );
		translucency = GetOpacityOfCompound( translucency, vinylsOpacity );

#endif
	DEPENDENT_CODE_END_AND( useVinyls, useVinylsBackside )

	DEPENDENT_CODE_START_AND( useVinyls, ! useVinylsBackside )
#if defined( _3DSMAX_ ) || ( defined( USE_VINYLS ) && ! defined( USE_VINYLS_BACKSIDE ) )

		vinylsOpacity = vinylsColour.a;

		diffuseTexColour.rgb = lerp( diffuseTexColour.rgb, vinylsColour.rgb, vinylsOpacity );
		translucency = GetOpacityOfCompound( translucency, vinylsOpacity );

		// Lower the intensity of cracks to 50% where covered by vinyls.
		vinylsCrackScaler = lerp( 1.0f, 0.5f, vinylsOpacity );

		// Change the intensity of specular and reflections / envmapping where covered by vinyls to the one provided by 'vinyl properties'.
		emapFactor = lerp( GLOBAL_ENVMAP_FACTOR, vinylsEMapFactor, vinylsOpacity );
		specularFactor = lerp( GLOBAL_SPECULAR_FACTOR, vinylsSpecularFactor, vinylsOpacity );

		// Vinyls have their own specular factor (constant).
		vinylsLightingSpecularFactor = specularFactor;

		// Force Fresnel control value to 1 on vinyls.
		fresnelControl = lerp( fresnelControl, 1.0f, vinylsOpacity );

		// Use white specular, after all, vinyls are plastic.
		vinylsEMapColour = lerp( specularTexColour, float4( 1.0f, 1.0f, 1.0f, 1.0f ), vinylsOpacity );

#endif
	DEPENDENT_CODE_END_AND( useVinyls, ! useVinylsBackside )

	//
	// Block (DIRT_SCRATCH):
	//   Based on values from control map, get the crack information (crackNormal, crackDistance),
	//   and apply dirt on the glass (modifies diffuseTexColour, translucency).
	//

#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
	// variable declarations
	float2 crackUV;
	float3 inputCrackNormal;
	float3 crackNormal;
	float crackDistance;
#endif

	DEPENDENT_CODE_START( dirtNscratch )
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )

	// Convert from Max's UV convention (inverted V).
	scratchControlUV.xy = Get2DUVCoordinates( _input.texCoord2.xy );
	controlColour = tex2D( scratchControlMap, scratchControlUV.xy );	// get control map colours 

	// get crack normal value allowing for scale and offset
	crackUV = (scratchControlUV.xy - crackUVoffset.xy) * crackUVoffset.zw;
	float4 inputCrackNormalColour = tex2D( crackMap, crackUV );	

	if ( specialCrackNormalMap )
	{
		inputCrackNormal = normalize(( inputCrackNormalColour.agb * 2.0f ) - 1.0f);	// special format version - alpha used to get better compression
	}
	else
	{
		inputCrackNormal = normalize(( inputCrackNormalColour.rgb * 2.0f ) - 1.0f);	// convert from colour space into normal map space
	}

	crackDistance = inputCrackNormal.z;							// keep the z coordinate for posterity - it gives a clue as to how far out we are (albeit reversed)
	// smoothstep the crack distance into something sensible when long range viewing results in mip-mapped normal maps
	crackDistance = 1.0f - smoothstep( 0.0f, 1.0f, (sensitivity - crackDistance) * rangeVisibility);

	crackNormal = float3( inputCrackNormal.x, inputCrackNormal.y, 0.00001f);					// clamp into plane of glass
	crackNormal = normalize( crackNormal );						// convert to true normal

	crackPercent = controlColour.g;
#if defined( _3DSMAX_ )
	if (testMode)
	{
		crackPercent = 1.0f;
	}
#endif	// _3DSMAX_

	// ramp the crack percent steeply so that it goes from nothing to a lot with a fairly small amount of damage
	crackPercent = smoothstep( 0.0f, 1.0f, (crackPercent - damageStartCracks) * damageCrackSlope);		// (this gives 0.0 for up to 0.25, then rapidly increasing to 1.0 by 0.35)
	// The cracks can be more or less hidden by the overlaid vinyls - so scale the percentage down.
	crackPercent *= vinylsCrackScaler;

	dirtPercent = controlColour.b;	// no messing with this one, its just as it is, a splotch of dirt, with variable thickness
	dirtOpacity = dirtPercent * (1.0f - translucency);

	// Apply dirt onto glass - colour, translucency, specular adjustment.
	translucency = 1.0f - (1.0f - dirtPercent) * (1.0f - translucency);	// output alpha needs to be scaled appropriately

	diffuseTexColour = lerp( diffuseTexColour, dirtBasis, dirtOpacity);

	// Scale down specular if there is any dirt on it
	dirtScratchSpecularModulation = smoothstep( 1.0f, 0.0f, dirtPercent * 4.0f );
#endif
	DEPENDENT_CODE_END( dirtNscratch )
	
	//
	// Block:
	//   Do the ambient lighting (store lit diffuseTexColour into accumulatedColour).
	//   (Note: The PSEUDO_GI path does not work, there is no normal yet.)
	//

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal_and_fresnel.xyz );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING ) || defined( DIRT_SCRATCH )
// declarations cannot be inside braces
	float3 tangent;
	float3 binormal;
#endif	// NORMAL_MAPPING || DIRT_SCRATCH

#if defined( PSEUDO_GI )

	// If reflection mapping is switched on
	float4 accumulatedColour;
	DEPENDENT_CODE_START( useReflectionMap )
#if defined( _3DSMAX_ ) || defined( REFLECTION_MAPPING )
		// Calculate ambient-lit base colour from interpolated lighting
		accumulatedColour = diffuseTexColour * _input.colour;
#endif
	DEPENDENT_CODE_ELSE( useReflectionMap )
#if defined( _3DSMAX_ ) || !defined( REFLECTION_MAPPING )
		// Calculate ambient-lit base colour (looking up the envmap for the ambient lighting)
		float4 envmapambient = texCUBElod( environmentMap, float4( normal.xyz, 4 ) );
		accumulatedColour = diffuseTexColour * envmapambient;
#endif
	DEPENDENT_CODE_END( useReflectionMap )

#else		// ! PSEUDO_GI
	float4 accumulatedColour = diffuseTexColour * _input.colour;
#endif	// ! PSEUDO_GI

	//
	// Block (NORMAL_MAPPING || DIRT_SCRATCH):
	//   Normalise tangent and binormal (interpolation makes them distort).
	//

	DEPENDENT_CODE_START_OR( useNormalMap, dirtNscratch )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING ) || defined( DIRT_SCRATCH )
		// Normalise the input tangent and binormal vectors
		tangent = normalize( _input.tangent );
		binormal = normalize( _input.binormal );
#endif
	DEPENDENT_CODE_END( useNormalMap )

	//
	// Block (NORMAL_MAPPING):
	//   Read and convert normal map (if used) - initializes the normal variable).
	//

	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord01.xy );

		float3 normalFromMap;
		if ( specialNormalMap )
		{
			normalFromMap = normalize( ( normalMapColour.agb * 2.0f ) - 1.0f );
		}
		else
		{
			normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );
		}
		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	//
	// Block:
	//   Calculate the Fresnel factor.
	//

	float fresnelCoefficient = lerp( 1.0f, saturate( _input.normal_and_fresnel.w ), fresnelControl );

	//
	// Block (DIRT_SCRATCH):
	//   Apply same perturbation to crack normal if used, and do crack-related environment mapping.
	//   (Adjusts accumulatedColour and translucency.)
	//

	float3 envMapSpaceNormal;
	float3 envMapSpaceEye;
	float3 reflectionVector;
	float4 environmentTexColour;

	DEPENDENT_CODE_START( dirtNscratch )
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
		// Perturb the tangent space normal by the normal map - not bothering with z as we set it to zero already, but the optimizer won't be sure (due to use of normalize)
		crackNormal = ( crackNormal.x * binormal ) + ( crackNormal.y * tangent );// + ( crackNormal.z * TSnormal );
		crackNormal = normalize( crackNormal );

		// if crackDistance is too great then there can be no crack output, because its in a flat region
		// again we use smoothstep to get rapid changes without causing aliasing errors
		crackDistance = smoothstep(0.0f, 1.0f, (crackLineScale - crackDistance) * crackStrength);	// now, if this is less than zero, there is no output
		
		float testDistance = crackDistance;
		
		testDistance *= (1.0f - dirtPercent);
		
		testDistance *= crackFinalScale;

		// Calculate the envmap space reflection vector
#if defined( _3DSMAX_ )
		// Max uses a map aligned to the object, always
		envMapSpaceNormal = mul( float4( crackNormal, 0.0f ), worldI ).xyz;
		envMapSpaceEye = mul( float4( -eye, 0.0f ), worldI ).xyz;
#else
		// The engine uses a map aligned to its own space, which may be object aligned, or to a parent object
		envMapSpaceNormal = mul( float4( crackNormal, 0.0f ), envTransformI ).xyz;
		envMapSpaceEye = mul( float4( -eye, 0.0f ), envTransformI ).xyz;
#endif	// ! _3DSMAX_
		reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		environmentTexColour = texCUBE( environmentMap, reflectionVector.xzy );
#else
		environmentTexColour = texCUBE( environmentMap, reflectionVector );
#endif

		// Calculate envmap colour and add to diffuse
		accumulatedColour += crackPercent * testDistance * crackColour * environmentTexColour * emapFactor;

		translucency = max( translucency, crackPercent * testDistance );

#endif
	DEPENDENT_CODE_END( dirtNscratch )

	//
	// Block (REFLECTION_MAPPING):
	//   If the shader requires reflection mapping, perform it.
	//   Adds reflection to accumulatedColour.
	//

	DEPENDENT_CODE_START( useReflectionMap )
#if defined( _3DSMAX_ ) || defined( REFLECTION_MAPPING )

		// Displace the reflection map lookup by an amount dependent on head movement and the
		// alpha channel of the current pixel
		float2 displacedTexCoord = _input.texCoord01.zw;

#if defined( ENABLE_PARRALAX_REFLECTION )
		float screenDistance = tex2D( reflectionMap, displacedTexCoord ).a;
		displacedTexCoord += float2( maxUOffset, maxVOffset ) * float2( headUMovement * screenDistance, headVMovement * screenDistance );
#endif

		// Do reflection mapping
		float4 reflectionTexColour = tex2D( reflectionMap, displacedTexCoord );

		// Calculate reflection map colour and add to diffuse
		accumulatedColour += vinylsEMapColour * reflectionTexColour * emapFactor * fresnelCoefficient;

		// Ensure that specular lighting is switched off for reflection-mapped materials (insides of
		// windscreens) as it's very expensive and not needed here.
		vinylsLightingSpecularFactor = 0.0f;

	//
	// Block (! REFLECTION_MAPPING):
	//   Otherwise (if not doing reflection mapping), do envmapping.
	//   Adds environment to accumulatedColour.

#endif
	DEPENDENT_CODE_ELSE( useReflectionMap )
#if defined( _3DSMAX_ ) || !defined( REFLECTION_MAPPING )

		// Calculate the envmap space reflection vector
#if defined( _3DSMAX_ )
		// Max uses a map aligned to the object, always
		envMapSpaceNormal = mul( float4( normal, 0.0f ), worldI ).xyz;
		envMapSpaceEye = mul( float4( -eye, 0.0f ), worldI ).xyz;
#else
		// The engine uses a map aligned to its own space, which may be object aligned, or to a parent object
		envMapSpaceNormal = mul( float4( normal, 0.0f ), envTransformI ).xyz;
		envMapSpaceEye = mul( float4( -eye, 0.0f ), envTransformI ).xyz;
#endif	// ! _3DSMAX_
		reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

#if defined( EXPERIMENTAL_ENVMAPPING )
	// Intersect the reflection vector with a 4m sphere centred on the car
	float3 envMapSpacePos = mul( float4( worldPos, 1.0f ), envTransformI ).xyz;
	float3 rayOrigin = envMapSpacePos;
	float3 rayDirection = reflectionVector;
	float B = dot( rayDirection, rayOrigin ) * 2.0f;
	float C = dot( rayOrigin, rayOrigin ) - ( 4.0f * 4.0f );
	float t0 = -B - sqrt( B*B - 4.0f * C ) / 2.0f;
	float t1 = -B + sqrt( B*B - 4.0f * C ) / 2.0f;
	float3 sphereIntersection = rayOrigin + ( rayDirection * max( t0, t1 ) );

	// Use the intersection point to look up the cubemap, not the reflection vector
	reflectionVector = normalize( sphereIntersection );
#endif

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour and add to diffuse
		accumulatedColour += dirtScratchSpecularModulation * vinylsEMapColour * environmentTexColour * emapFactor * fresnelCoefficient;
#endif
	DEPENDENT_CODE_END( useReflectionMap )

	//
	// Block (USE_VINYLS)
	//   Adjust the specular properties just before we compute per-pixel lighting.
	//   The specular is "exclusively" for vinyls, thus we do not have to do any lerps or other computations -
	//   vinylsLightSpecularFactor is the factor that properly scales the per-pixel specular contribution
	//   depending on the opacity of the vinyl layer.
	//

	DEPENDENT_CODE_START_AND( useVinyls, ! useVinylsBackside )
#if defined( _3DSMAX_ ) || ( defined( USE_VINYLS ) && ! defined( USE_VINYLS_BACKSIDE ) )

		specularTexColour = vinylsEMapColour;

		// Feed the user-supplied specular power where the per-pixel lighting infrastructure expects it.
		specularTexColour.a = vinylsSpecularPower;

#endif
	DEPENDENT_CODE_END_AND( useVinyls, useVinylsBackside )

	// We use global specular factor. Scale the specular factor down in presence of dirt.
	float globalSpecularFactorValue = vinylsLightingSpecularFactor * dirtScratchSpecularModulation;

	//
	// Perform per-pixel lighting (including shadows).
	//

	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = translucency;
//	accumulatedColour *= broken;
	
#ifdef SPECIAL_WRAPPING
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
//	float2 adjust = float2( int(crackUV.x), int(crackUV.y));
//	adjust += 4.0f;
//	adjust *= 0.03125f;
//	accumulatedColour.xy += adjust;

	float2 adjust = crackUV - 0.5f;
	adjust = abs( adjust );
	float mmx = max( adjust.x, adjust.y );
	if (mmx > 0.5f)
	{
		accumulatedColour.xy = adjust.xy;
		accumulatedColour.z = ( ( sign(adjust.x) * sign(adjust.y) ) + 1.0f) * 0.5f;
		accumulatedColour.w = 1.0f;
	}
	
	float2 temp = abs(crackUV);
	float2 line = temp - float2( int(temp.x), int(temp.y));
	if (line.x < 0.05f) accumulatedColour.z = 1.0f;
	if (line.y < 0.05f) accumulatedColour.z = 1.0f;
	
#endif	
#endif	// SPECIAL_WRAPPING

#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
// quick test
	//accumulatedColour.ga += crackPercent * 0.5f;
#endif

	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}




//
// Low Detail Shaders
//


struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD2;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
#if defined( _3DSMAX_ ) || !defined( REFLECTION_MAPPING )
	float4 reflectionVector_plus_fresnel	: TEXCOORD1;		// Reflection vector (envmap space), with fresnel term in w
#endif
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
	float2 texCoord2	: TEXCOORD3;												// UV coords for vinyls.
#endif
};


VSOUTPUT_LD GlassLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord0;

	DEPENDENT_CODE_START( useVinyls )
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
		_output.texCoord2 = _input.texCoord2;
#endif
	DEPENDENT_CODE_END( useVinyls )

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// get normal in world space and do lighting
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	float3 eye = normalize( worldEyeVec );

	DEPENDENT_CODE_START( useReflectionMap )
	DEPENDENT_CODE_ELSE( useReflectionMap )
#if defined( _3DSMAX_ ) || !defined( REFLECTION_MAPPING )

		// Calculate the envmap space reflection vector
#if defined( _3DSMAX_ )
		// Max uses a map aligned to the object, always
		float3 envMapSpaceNormal = mul( float4( normal.xyz, 0.0f ), worldI ).xyz;
		float3 envMapSpaceEye = mul( -eye, worldI ).xyz;
#else
		// The engine uses a map aligned to its own space, which may be object aligned, or to a parent object
		float3 envMapSpaceNormal = mul( float4( normal.xyz, 0.0f ), envTransformI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), envTransformI ).xyz;
#endif
		float3 reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

		// Calculate fresnel term
		_output.reflectionVector_plus_fresnel = float4( reflectionVector, calculated_fresnel( -eye, normal, fresnelFactor ) );

#endif
	DEPENDENT_CODE_END( useReflectionMap )

#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	// keep alpha value from the vertex
#if defined( _3DSMAX_ )
	_output.colour.a = _input.alpha.r;
#else
	_output.colour.a = _input.colour.a;
#endif

	return _output;
}


PSOUTPUT GlassLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 diffuseTexColour;
	DEPENDENT_CODE_START( useConstantDiffuse )
#if defined( _3DSMAX_ ) || defined( USE_CONSTANT_DIFFUSE )
		diffuseTexColour  = constantDiffuse;
#endif
	DEPENDENT_CODE_ELSE( useConstantDiffuse )
#if defined( _3DSMAX_ ) || ! defined( USE_CONSTANT_DIFFUSE )
		diffuseTexColour  = tex2D( diffuseMap, _input.texCoord );
#endif
	DEPENDENT_CODE_END( useConstantDiffuse )

#if defined( _3DSMAX_ ) || defined( SPECULAR_MAPPING )
	float4 specularTexColour = tex2D( specularMap, _input.texCoord );
#else
	float4 specularTexColour = float4( 1.0f, 1.0f, 1.0f, 1.0f );
#endif

#if defined( _3DSMAX_ ) || defined( FRESNEL_MAPPING )
	float  fresnelControl = tex2D( fresnelMap, _input.texCoord ).g;
#else
	float  fresnelControl = 1.0f;
#endif

#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
	float4 vinylsColour = tex2D( vinylsMap, Get2DUVCoordinates( _input.texCoord2.xy ) );
#endif

	// Factor vertex alpha into the diffuse alpha
	float translucency = diffuseTexColour.a * _input.colour.a;

	// Premultiply the diffuse colour by the translucency
	diffuseTexColour.rgb *= translucency;

	float emapFactor = GLOBAL_ENVMAP_FACTOR;			// On pure glass, reflections / envmapping at full intensity are controlled by glass properties.

	DEPENDENT_CODE_START( useVinyls )
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
		diffuseTexColour.rgb = lerp( diffuseTexColour.rgb, vinylsColour.rgb, vinylsColour.a );
		translucency = GetOpacityOfCompound( translucency, vinylsColour.a );
#endif
	DEPENDENT_CODE_END( useVinyls )

	DEPENDENT_CODE_START_AND( useVinyls, ! useVinylsBackside )
#if defined( _3DSMAX_ ) || ( defined( USE_VINYLS ) && ! defined( USE_VINYLS_BACKSIDE ) )
		// Change the intensity of reflections / envmapping where covered by vinyls.
		emapFactor = lerp( GLOBAL_ENVMAP_FACTOR, vinylsEMapFactor, vinylsColour.a );
#endif
	DEPENDENT_CODE_END_AND( useVinyls, ! useVinylsBackside )

	// Apply vertex lighting
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	DEPENDENT_CODE_START( useReflectionMap )
	DEPENDENT_CODE_ELSE( useReflectionMap )
#if defined( _3DSMAX_ ) || !defined( REFLECTION_MAPPING )

		// Calculate the fresnel factor
		float fresnelCoefficient = lerp( 1.0f, saturate( _input.reflectionVector_plus_fresnel.w ), fresnelControl );

		// Apply envmapping
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBE( environmentMap, _input.reflectionVector_plus_fresnel.xzy );
#else
		float4 environmentTexColour = texCUBE( environmentMap, _input.reflectionVector_plus_fresnel.xyz );
#endif

		// Calculate envmap colour and add to diffuse
		accumulatedColour += environmentTexColour * specularTexColour * emapFactor * fresnelCoefficient;

#endif
	DEPENDENT_CODE_END( useReflectionMap )

	accumulatedColour.a = translucency;
	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );

	return _output;
}




//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Glass
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Glass";
	int    normalDeferredID			= 2;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Glass_LowDetail";
	int    lowDetailDeferredID = 2;
#if !defined( REFLECTION_MAPPING )
	bool   usefullprecision = true;
#endif
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx GlassVertexShader();
		PixelShader = compile sce_fp_rsx GlassFragmentShader();
#else
		VertexShader = compile vs_3_0 GlassVertexShader();
		PixelShader = compile ps_3_0 GlassFragmentShader();
#endif
	}
}



technique Glass_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Glass_LowDetail";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx GlassLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx GlassLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 GlassLowDetailVertexShader();
		PixelShader = compile ps_3_0 GlassLowDetailFragmentShader();
#endif
	}
}
